package com.nicbase.boxdata;

import java.util.HashMap;
import java.util.Map;

import lombok.Getter;
import lombok.ToString;

import com.nicbase.tools.serialisation.json.ICustomizedJSONEnum;

/**
 * This class represents the Errors of a sending box.
 *
 * @author ps
 * @author MBausch
 */
@ToString
public enum Error implements ICustomizedJSONEnum {

    DO_NOT_USE(0, Error.CATEGORY_NONE, 1, "DO_NOT_USE"),

    NO_GPS(1, Error.CATEGORY_NONE, 2, "NO_GPS"),

    LOW_BATTERY(2, Error.CATEGORY_NONE, 3, "LOW_BATTERY"),

    VANLOXX_COM_ERROR(3, Error.CATEGORY_VANLOXX, 1, "VANLOXX_COM_ERROR"),

    VANLOXX_NO_ANSWER(4, Error.CATEGORY_VANLOXX, 2, "VANLOXX_NO_ANSWER"),

    VANLOXX_CRC_ERROR(5, Error.CATEGORY_VANLOXX, 3, "VANLOXX_CRC_ERROR"),

    VANLOXX_WRONG_SERIAL(6, Error.CATEGORY_VANLOXX, 4, "VANLOXX_WRONG_SERIAL"),

    VANLOXX_FREQUENT_CRC_ERROR(7, Error.CATEGORY_VANLOXX, 5, "VANLOXX_FREQUENT_CRC_ERROR"),

    /**
     * Hardware-Komponente nicht erreichbar.
     */
    KRONE_ERROR_SYSTEM_NODEVICEFOUND(8, Error.CATEGORY_KRONE, 1, "KRONE_ERROR_SYSTEM_NODEVICEFOUND"),
    /**
     * Kommunikation mit Hardware Komponente nicht möglich.
     */
    KRONE_ERROR_SYSTEM_DEVICECOMMUNICATION(9, Error.CATEGORY_KRONE, 2, "KRONE_ERROR_SYSTEM_DEVICECOMMUNICATION"),
    /**
     * Konfiguration falsch, fehlerhaft, fehlend.
     */
    KRONE_ERROR_SYSTEM_CONFIGURATION(10, Error.CATEGORY_KRONE, 3, "KRONE_ERROR_SYSTEM_CONFIGURATION"),
    /**
     * Betriebstemperatur ausserhalb des Intervalls.
     */
    KRONE_ERROR_SYSTEM_TEMPERATURE(11, Error.CATEGORY_KRONE, 4, "KRONE_ERROR_SYSTEM_TEMPERATURE"),
    /**
     * Problem bei Initialisierung/Programmstart.
     */
    KRONE_ERROR_GPS_DAEMONINIT(12, Error.CATEGORY_KRONE, 5, "KRONE_ERROR_GPS_DAEMONINIT"),
    /**
     * GPS Modul kann nicht angesprochen werden.
     */
    KRONE_ERROR_GPS_NODEVICE(13, Error.CATEGORY_KRONE, 6, "KRONE_ERROR_GPS_NODEVICE"),
    /**
     * GPS Modul gefunden, aber Fehler in Kommunikation.
     */
    KRONE_ERROR_GPS_COMMUNICATION(14, Error.CATEGORY_KRONE, 7, "KRONE_ERROR_GPS_COMMUNICATION"),
    /**
     * Konfiguration falsch, fehlerhaft, fehlend.
     */
    KRONE_ERROR_GPS_CONFIGURATION(15, Error.CATEGORY_KRONE, 8, "KRONE_ERROR_GPS_CONFIGURATION"),
    /**
     * Problem bei Initialisierung/Programmstart.
     */
    KRONE_ERROR_EBSCLIENT_DAEMONINIT(16, Error.CATEGORY_KRONE, 9, "KRONE_ERROR_EBSCLIENT_DAEMONINIT"),
    /**
     * Keine Verbindung von EBS Client zum CAN.
     */
    KRONE_ERROR_EBSCLIENT_NOCAN(17, Error.CATEGORY_KRONE, 10, "KRONE_ERROR_EBSCLIENT_NOCAN"),
    /**
     * Konfiguration falsch, fehlerhaft, fehlend.
     */
    KRONE_ERROR_EBSCLIENT_CONFIGURATION(18, Error.CATEGORY_KRONE, 11, "KRONE_ERROR_EBSCLIENT_CONFIGURATION"),
    /**
     * Problem bei Initialisierung/Programmstart.
     */
    KRONE_ERROR_BATTERYMANAGER_DAEMONINIT(19, Error.CATEGORY_KRONE, 12, "KRONE_ERROR_BATTERYMANAGER_DAEMONINIT"),
    /**
     * zu schnelle Entladung oder anderes Fehlverhalten der Batterie.
     */
    KRONE_ERROR_BATTERYMANAGER_BATTERYDAMAGED(20, Error.CATEGORY_KRONE, 13, "KRONE_ERROR_BATTERYMANAGER_BATTERYDAMAGED"),
    /**
     * Problem bei Initialisierung/Programmstart.
     */
    KRONE_ERROR_TEMPERATURED_DAEMONINIT(21, Error.CATEGORY_KRONE, 14, "KRONE_ERROR_TEMPERATURED_DAEMONINIT"),
    /**
     * unzureichender Speicherplatz für die Temperaturwerte.
     */
    KRONE_ERROR_TEMPERATURED_NOSPACE(22, Error.CATEGORY_KRONE, 15, "KRONE_ERROR_TEMPERATURED_NOSPACE"),
    /**
     * Konfiguration falsch, fehlerhaft, fehlend.
     */
    KRONE_ERROR_TEMPERATURED_CONFIGURATION(23, Error.CATEGORY_KRONE, 16, "KRONE_ERROR_TEMPERATURED_CONFIGURATION"),
    /**
     * keine Verbindung zum 1wire Bus.
     */
    KRONE_ERROR_TEMPERATURED_NOONEWIRECONN(24, Error.CATEGORY_KRONE, 17, "KRONE_ERROR_TEMPERATURED_NOONEWIRECONN"),
    /**
     * Kurzschluss auf 1wire Bus.
     */
    KRONE_ERROR_TEMPERATURED_ONEWIRESHORTCIRCUIT(25, Error.CATEGORY_KRONE, 18, "KRONE_ERROR_TEMPERATURED_ONEWIRESHORTCIRCUIT"),
    /**
     * Fehlender Temperatursensor erkannt.
     */
    KRONE_ERROR_TEMPERATURED_MISSINGSENSOR(26, Error.CATEGORY_KRONE, 19, "KRONE_ERROR_TEMPERATURED_MISSINGSENSOR"),
    /**
     * Problem bei Initialisierung/Programmstart.
     */
    KRONE_ERROR_COOLINGD_DAEMONINIT(27, Error.CATEGORY_KRONE, 20, "KRONE_ERROR_COOLINGD_DAEMONINIT"),
    /**
     * keine Verbindung zu Kühlanlage.
     */
    KRONE_ERROR_COOLINGD_NOORUNKNOWNDEVICE(28, Error.CATEGORY_KRONE, 21, "KRONE_ERROR_COOLINGD_NOORUNKNOWNDEVICE"),
    /**
     * Problem bei Initialisierung/Programmstart.
     */
    KRONE_ERROR_DOORLOCKD_DAEMONINIT(29, Error.CATEGORY_KRONE, 22, "KRONE_ERROR_DOORLOCKD_DAEMONINIT"),
    /**
     * keine Verbindung zu VanLOXX Steuergerät.
     */
    KRONE_ERROR_DOORLOCKD_NOORUNKNOWNDEVICE(30, Error.CATEGORY_KRONE, 23, "KRONE_ERROR_DOORLOCKD_NOORUNKNOWNDEVICE"),
    /**
     * Konfiguration falsch, fehlerhaft, fehlend.
     */
    KRONE_ERROR_DOORLOCKD_CONFIGURATION(31, Error.CATEGORY_KRONE, 24, "KRONE_ERROR_DOORLOCKD_CONFIGURATION"),
    /**
     * Problem bei Initialisierung/Programmstart.
     */
    KRONE_ERROR_COUPLINGD_DAEMONINIT(32, Error.CATEGORY_KRONE, 25, "KRONE_ERROR_COUPLINGD_DAEMONINIT"),
    /**
     * keine Verbindung zu ADU Hardware.
     */
    KRONE_ERROR_COUPLINGD_NODEVICE(33, Error.CATEGORY_KRONE, 26, "KRONE_ERROR_COUPLINGD_NODEVICE"),
    /**
     * Konfiguration falsch, fehlerhaft, fehlend.
     */
    KRONE_ERROR_COUPLINGD_CONFIGURATION(34, Error.CATEGORY_KRONE, 27, "KRONE_ERROR_COUPLINGD_CONFIGURATION"),
    /**
     * Problem bei Initialisierung/Programmstart.
     */
    KRONE_ERROR_CONNECTIONMANAGER_DAEMONINIT(35, Error.CATEGORY_KRONE, 28, "KRONE_ERROR_CONNECTIONMANAGER_DAEMONINIT"),
    /**
     * keine Verbindung zu GSM Chip.
     */
    KRONE_ERROR_CONNECTIONMANAGER_NODEVICE(36, Error.CATEGORY_KRONE, 29, "KRONE_ERROR_CONNECTIONMANAGER_NODEVICE"),
    /**
     * keine Simkarte erkannt.
     */
    KRONE_ERROR_CONNECTIONMANAGER_NOSIM(37, Error.CATEGORY_KRONE, 30, "KRONE_ERROR_CONNECTIONMANAGER_NOSIM"),
    /**
     * falsche Simkarten-PIN.
     */
    KRONE_ERROR_CONNECTIONMANAGER_WRONGPIN(38, Error.CATEGORY_KRONE, 31, "KRONE_ERROR_CONNECTIONMANAGER_WRONGPIN"),
    /**
     * Problem bei Initialisierung/Programmstart.
     */
    KRONE_ERROR_LEDD_DAEMONINIT(39, Error.CATEGORY_KRONE, 32, "KRONE_ERROR_LEDD_DAEMONINIT"),
    /**
     * keine Verbindung zu LEDs kann.
     */
    KRONE_ERROR_LEDD_NODEVICE(40, Error.CATEGORY_KRONE, 33, "KRONE_ERROR_LEDD_NODEVICE"),
    /**
     * Konfiguration falsch, fehlerhaft, fehlend.
     */
    KRONE_ERROR_LEDD_CONFIGURATION(41, Error.CATEGORY_KRONE, 34, "KRONE_ERROR_LEDD_CONFIGURATION"),
    /**
     * Problem bei Initialisierung/Programmstart.
     */
    KRONE_ERROR_ISOBUSFSD_DAEMONINIT(42, Error.CATEGORY_KRONE, 35, "KRONE_ERROR_ISOBUSFSD_DAEMONINIT"),
    /**
     * Konfiguration falsch, fehlerhaft, fehlend.
     */
    KRONE_ERROR_ISOBUSFSD_CONFIGURATION(43, Error.CATEGORY_KRONE, 36, "KRONE_ERROR_ISOBUSFSD_CONFIGURATION"),
    /**
     * Problem bei Initialisierung/Programmstart.
     */
    KRONE_ERROR_PDUD_DAEMONINIT(44, Error.CATEGORY_KRONE, 37, "KRONE_ERROR_PDUD_DAEMONINIT"),
    /**
     * Konfiguration falsch, fehlerhaft, fehlend.
     */
    KRONE_ERROR_PDUD_CONFIGURATION(45, Error.CATEGORY_KRONE, 38, "KRONE_ERROR_PDUD_CONFIGURATION"),
    /**
     * Problem bei Initialisierung/Programmstart.
     */
    KRONE_ERROR_VPNCLIENT_DAEMONINIT(46, Error.CATEGORY_KRONE, 39, "KRONE_ERROR_VPNCLIENT_DAEMONINIT"),
    /**
     * Konfiguration falsch, fehlerhaft, fehlend.
     */
    KRONE_ERROR_VPNCLIENT_CONFIGURATION(47, Error.CATEGORY_KRONE, 40, "KRONE_ERROR_VPNCLIENT_CONFIGURATION"),
    /**
     * Problem bei Initialisierung/Programmstart.
     */
    KRONE_ERROR_NICBASECLIENT_DAEMONINIT(48, Error.CATEGORY_KRONE, 41, "KRONE_ERROR_NICBASECLIENT_DAEMONINIT"),
    /**
     * Konfiguration falsch, fehlerhaft, fehlend.
     */
    KRONE_ERROR_NICBASECLIENT_CONFIGURATION(49, Error.CATEGORY_KRONE, 42, "KRONE_ERROR_NICBASECLIENT_CONFIGURATION"),
    /**
     * Problem bei Initialisierung/Programmstart.
     */
    KRONE_ERROR_FARMPILOTCLIENT_DAEMONINIT(50, Error.CATEGORY_KRONE, 43, "KRONE_ERROR_FARMPILOTCLIENT_DAEMONINIT"),
    /**
     * Konfiguration falsch, fehlerhaft, fehlend.
     */
    KRONE_ERROR_FARMPILOTCLIENT_CONFIGURATION(51, Error.CATEGORY_KRONE, 44, "KRONE_ERROR_FARMPILOTCLIENT_CONFIGURATION"),
    /**
     * Problem bei Initialisierung/Programmstart.
     */
    KRONE_ERROR_TELEMATICSCLIENT_DAEMONINIT(52, Error.CATEGORY_KRONE, 45, "KRONE_ERROR_TELEMATICSCLIENT_DAEMONINIT"),
    /**
     * Konfiguration falsch, fehlerhaft, fehlend.
     */
    KRONE_ERROR_TELEMATICSCLIENT_CONFIGURATION(53, Error.CATEGORY_KRONE, 46, "KRONE_ERROR_TELEMATICSCLIENT_CONFIGURATION"),
    /**
     * Problem bei Initialisierung/Programmstart.
     */
    KRONE_ERROR_NTRIPCLIENT_DAEMONINIT(54, Error.CATEGORY_KRONE, 47, "KRONE_ERROR_NTRIPCLIENT_DAEMONINIT"),
    /**
     * Konfiguration falsch, fehlerhaft, fehlend.
     */
    KRONE_ERROR_NTRIPCLIENT_CONFIGURATION(55, Error.CATEGORY_KRONE, 48, "KRONE_ERROR_NTRIPCLIENT_CONFIGURATION"),
    /**
     * Problem bei Initialisierung/Programmstart.
     */
    KRONE_ERROR_WLAN2CAN_DAEMONINIT(56, Error.CATEGORY_KRONE, 49, "KRONE_ERROR_WLAN2CAN_DAEMONINIT"),
    /**
     * Konfiguration falsch, fehlerhaft, fehlend.
     */
    KRONE_ERROR_WLAN2CAN_CONFIGURATION(57, Error.CATEGORY_KRONE, 50, "KRONE_ERROR_WLAN2CAN_CONFIGURATION"),
    /**
     * Problem bei Initialisierung/Programmstart.
     */
    KRONE_ERROR_ECUD_DAEMONINIT(58, Error.CATEGORY_KRONE, 51, "KRONE_ERROR_ECUD_DAEMONINIT"),
    /**
     * Konfiguration falsch, fehlerhaft, fehlend.
     */
    KRONE_ERROR_ECUD_CONFIGURATION(59, Error.CATEGORY_KRONE, 52, "KRONE_ERROR_ECUD_CONFIGURATION"),
    /**
     * Problem bei Initialisierung/Programmstart.
     */
    KRONE_ERROR_TCLOGD_DAEMONINIT(60, Error.CATEGORY_KRONE, 53, "KRONE_ERROR_TCLOGD_DAEMONINIT"),
    /**
     * Konfiguration falsch, fehlerhaft, fehlend.
     */
    KRONE_ERROR_TCLOGD_CONFIGURATION(61, Error.CATEGORY_KRONE, 54, "KRONE_ERROR_TCLOGD_CONFIGURATION"),
    /**
     * Problem bei Initialisierung/Programmstart.
     */
    KRONE_ERROR_TCEXPORTCLIENT_DAEMONINIT(62, Error.CATEGORY_KRONE, 55, "KRONE_ERROR_TCEXPORTCLIENT_DAEMONINIT"),
    /**
     * Konfiguration falsch, fehlerhaft, fehlend.
     */
    KRONE_ERROR_TCEXPORTCLIENT_CONFIGURATION(63, Error.CATEGORY_KRONE, 56, "KRONE_ERROR_TCEXPORTCLIENT_CONFIGURATION"),
    /**
     * Problem bei Initialisierung/Programmstart.
     */
    KRONE_ERROR_CANLOGD_DAEMONINIT(64, Error.CATEGORY_KRONE, 57, "KRONE_ERROR_CANLOGD_DAEMONINIT"),
    /**
     * Konfiguration falsch, fehlerhaft, fehlend.
     */
    KRONE_ERROR_CANLOGD_CONFIGURATION(65, Error.CATEGORY_KRONE, 58, "KRONE_ERROR_CANLOGD_CONFIGURATION"),

    HAS_DIAGNOSTIC_ERROR_CODE(66, Error.CATEGORY_NONE, 4, "HAS_DIAGNOSTIC_ERROR_CODE"),

    ERROR_COOLINGD_NOCONNECTIONTOFUELSENSOR(67, Error.CATEGORY_KRONE, 59, "ERROR_COOLINGD_NOCONNECTIONTOFUELSENSOR"),

    ERROR_COOLINGD_NOCONNECTIONTOFLOATSENSOR(68, Error.CATEGORY_KRONE, 60, "ERROR_COOLINGD_NOCONNECTIONTOFLOATSENSOR"),

    ERROR_COOLINGD_NOCONNECTIONTOFUELGAUGE(69, Error.CATEGORY_KRONE, 61, "ERROR_COOLINGD_NOCONNECTIONTOFUELGAUGE");

    public static final String CATEGORY_NONE    = "NONE";
    public static final String CATEGORY_KRONE   = "KRONE";
    public static final String CATEGORY_VANLOXX = "VANLOXX";

    /**
     * Map for indexed access to Error via internal Id
     */
    private static Map<Integer, Error> internalIdMap = new HashMap<>();
    /**
     * Map for indexed access to Error via external Id
     */
    private static Map<String, Error>  externalIdMap = new HashMap<>();

    /**
     * Id to identify one enum without ordinal.
     */
    @Getter
    private int    internalId;
    /**
     * Id to identify one enum via mapping form external
     */
    @Getter
    private String externalId;
    /**
     * Id for the translation of one enum
     */
    @Getter
    private String translationId;

    /**
     * Constructor with external int id
     *
     * @param internalId
     * @param category
     * @param externalId
     * @param translationId
     */
    private Error(final int internalId, final String category, final int externalId, final String translationId) {
        this(internalId, category + "_" + String.valueOf(externalId), translationId);
    }

    /**
     * Constructor with all arguments
     *
     * @param internalId
     * @param externalId
     * @param translationId
     */
    private Error(final int internalId, final String externalId, final String translationId) {
        this.internalId = internalId;
        this.externalId = externalId;
        this.translationId = "error." + translationId.trim();
    }

    /*
     * (non-Javadoc)
     *
     * @see com.nicbase.tools.serialisation.json.ICustomizedJSONEnum#getDatabaseRepresentation()
     */
    @Override
    public Object getDatabaseRepresentation() {
        // no escaping needed
        return getInternalId();
    }

    /**
     * This method returns one erro via internal Identifier
     *
     * @param internalId
     * @return Error or null if ordinal is not set
     */
    public static Error fromInternalId(final int internalId) {
        // fill internal Id Map
        if (internalIdMap.isEmpty()) {
            for (Error error : Error.values()) {
                internalIdMap.put(error.getInternalId(), error);
            }
        }
        return internalIdMap.get(internalId);
    }

    /**
     * This method returns one erro via externalId Identifier
     *
     * @param category
     * @param externalId
     * @return Error or null if ordinal is not set
     */
    public static Error fromExternalId(final String category, final int externalId) {
        return fromExternalId(category + "_" + String.valueOf(externalId));
    }

    /**
     * This method returns one erro via externalId Identifier
     *
     * @param externalId
     * @return Error or null if ordinal is not set
     */
    public static Error fromExternalId(final String externalId) {
        // fill internal Id Map
        if (externalIdMap.isEmpty()) {
            for (Error error : Error.values()) {
                externalIdMap.put(error.getExternalId(), error);
            }
        }

        return externalIdMap.get(externalId);
    }

}
